//  2011 IDesign Inc.
// Pytania? Komentarze? Odwied
// http://www.idesign.net

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Security.Permissions;
using System.Threading;

namespace ServiceModelEx
{
   [SecurityPermission(SecurityAction.Demand,ControlThread = true)]
   public class ThreadPoolSynchronizer : SynchronizationContext,IDisposable
   {
      class WorkerThread
      {
         ThreadPoolSynchronizer m_Context;
         public Thread m_ThreadObj;
         bool m_EndLoop;

         public int ManagedThreadId
         {
            get
            {
               return m_ThreadObj.ManagedThreadId;
            }
         }

         internal WorkerThread(string name,ThreadPoolSynchronizer context)
         {
            m_Context = context;

            m_EndLoop = false;
            m_ThreadObj = null;

            m_ThreadObj = new Thread(Run);
            m_ThreadObj.IsBackground = true;
            m_ThreadObj.Name = name;
            m_ThreadObj.Start();
         }
         bool EndLoop
         {
            set
            {
               lock(this)
               {
                  m_EndLoop = value;
               }
            }
            get
            {
               lock(this)
               {
                  return m_EndLoop;
               }
            }
         }
         void Start()
         {
            Debug.Assert(m_ThreadObj != null);
            Debug.Assert(m_ThreadObj.IsAlive == false);
            m_ThreadObj.Start();
         }
         void Run()
         {
            Debug.Assert(SynchronizationContext.Current == null);
            SynchronizationContext.SetSynchronizationContext(m_Context);

            while(EndLoop == false)
            {
               WorkItem workItem = m_Context.GetNext();
               if(workItem != null)
               {
                  workItem.CallBack();
               }
            }
         }
         public void Kill()
         {
            // Metoda wywoywana dla wtku klienta (musi uy buforowanego obiektu wtku)
            Debug.Assert(m_ThreadObj != null);
            if(m_ThreadObj.IsAlive == false)
            {
               return;
            }
            EndLoop = true;

            // Czeka na zabicie wtku
            m_ThreadObj.Join();
         }
      }

      WorkerThread[] m_WorkerThreads;
      Queue<WorkItem> m_WorkItemQueue;
      protected Semaphore CallQueued
      {get;private set;}
 
      public ThreadPoolSynchronizer(uint poolSize) : this(poolSize,"Wtek w puli: ")
      {}

      public ThreadPoolSynchronizer(uint poolSize,string poolName)
      {
         if(poolSize == 0)
         {
            throw new InvalidOperationException("Rozmiar puli musi by rny od zera");
         }
         CallQueued = new Semaphore(0,Int32.MaxValue);
         m_WorkItemQueue = new Queue<WorkItem>();

         m_WorkerThreads = new WorkerThread[poolSize];
         for(int index = 0;index<poolSize;index++)
         {
            m_WorkerThreads[index] = new WorkerThread(poolName + " " + (index+1),this);
         }
      }
      virtual internal void QueueWorkItem(WorkItem workItem)
      {
         lock(m_WorkItemQueue)
         {
            m_WorkItemQueue.Enqueue(workItem);
            CallQueued.Release();
         }
      }
      protected virtual bool QueueEmpty
      {
         get
         {
            lock(m_WorkItemQueue)
            {
               if(m_WorkItemQueue.Count > 0)
               {
                  return false;
               }
               return true;
            }
         }
      }
      internal virtual WorkItem GetNext()
      {
         CallQueued.WaitOne();
         lock(m_WorkItemQueue)
         {
            if(m_WorkItemQueue.Count == 0)
            {
               return null;
            }
            return m_WorkItemQueue.Dequeue();
         }
      }
      public void Dispose()
      {
         Close();
      }
      public override SynchronizationContext CreateCopy()
      {
         return this;
      }
      public override void Post(SendOrPostCallback method,object state)
      {
         WorkItem workItem = new WorkItem(method,state);
         QueueWorkItem(workItem);
      }
      public override void Send(SendOrPostCallback method,object state)
      {
         // Jeli kontekst jest ju prawidowy, wywoanie pozwala uniknc zakleszczenia
         if(SynchronizationContext.Current == this)
         {
            method(state);
            return;
         }
         WorkItem workItem = new WorkItem(method,state);
         QueueWorkItem(workItem);
         workItem.AsyncWaitHandle.WaitOne();
      }
      public void Close()
      {
         if(CallQueued.SafeWaitHandle.IsClosed)
         {
            return;
         }
         CallQueued.Release(Int32.MaxValue);

         foreach(WorkerThread thread in m_WorkerThreads)
         {
            thread.Kill();
         }
         CallQueued.Close();
      }
      public void Abort()
      {
         CallQueued.Release(Int32.MaxValue);

         foreach(WorkerThread thread in m_WorkerThreads)
         {
            thread.m_ThreadObj.Abort();
         }
         CallQueued.Close();
      }
   }
}